#include <stdlib.h>             /* For exit */
#include <stdio.h>              /* For printf */
#include <string.h>             /* For strcmp */
#include <math.h>               /* For sqrt */
#include "e1432.h"
#include "xplot.h"

/* #define OTF_SPAN_DOWN */ /* uncomment to change span on the fly */
#define	OTF_SPAN_DOWN
/* uncomment to change ac coupling frequency or autozero offset on the fly */
/* #define OTF_COUPLE_FREQ_AZ_TOGGLE */
/* #ifdef	OTF_RANGE_TOGGLE */ /* uncomment to toggle range on the fly */

/* These probably belong in e1432.h, but they're not there now */
extern SHORTSIZ16 EXPORT
e1432_get_cal_failures(E1432ID hw, SHORTSIZ16 ID, LONGSIZ32 * value);
extern SHORTSIZ16 EXPORT
e1432_get_cal_gain(E1432ID hw, SHORTSIZ16 ID, FLOATSIZ32 * value);
extern SHORTSIZ16 EXPORT
e1432_get_cal_offset(E1432ID hw, SHORTSIZ16 ID, FLOATSIZ32 * value);

#define SIZE 16384
#define MAX_LAS		8
#define MAX_CHAN	(MAX_LAS * E1432_INPUT_CHANS)
#define MAX_PLOT_SIZE	1024

/* Wrap this around all the many function calls which might fail */
#define DEBUG(s)        s
#ifdef  __lint
#define CHECK(func)     \
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
        DEBUG((void) printf("Error: %s returned %d\n", #func, _s));\
        return _s;\
    }\
} while (func)
#else
#define CHECK(func)     \
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
        DEBUG((void) printf("Error: %s returned %d\n", #func, _s));\
        return _s;\
    }\
} while (0)
#endif

#define WIDTH           500
#define HEIGHT          400

SHORTSIZ16 data_size_enums[] =
{
    E1432_DATA_SIZE_16,
    E1432_DATA_SIZE_32_SERV, E1432_DATA_SIZE_32_SERV,
    E1432_DATA_SIZE_FLOAT32, E1432_DATA_SIZE_FLOAT32,
    E1432_DATA_SIZE_32  /* last so will not also match 32s, etc */
};
char *data_size_strs[] =
{
    "16",
    "32s", "32S",
    "f32", "F32",
    "32",
    NULL
};

SHORTSIZ16 data_mode_enums[] =
{
    E1432_BLOCK_MODE,                E1432_BLOCK_MODE,
    E1432_CONTINUOUS_MODE,           E1432_CONTINUOUS_MODE,
    E1432_DATA_MODE_OVERLAP_FREERUN, E1432_DATA_MODE_OVERLAP_FREERUN,
    E1432_DATA_MODE_OVERLAP_BLOCK,   E1432_DATA_MODE_OVERLAP_BLOCK };
char *data_mode_strs[] =
{
    "b", "B",
    "c", "C",
    "f", "F",
    "o", "O",
    NULL
};


FLOATSIZ32 range = 10.0;    /* default range */


SHORTSIZ16 print_lims(E1432ID hw, SHORTSIZ16 group,
 SHORTSIZ16 (*plimf)(E1432ID, SHORTSIZ16, FLOATSIZ32 *, FLOATSIZ32 *,
 FLOATSIZ32 *, FLOATSIZ32 *), char *pname)
{
    FLOATSIZ32 min, max, def, step;
    SHORTSIZ16 err;

    err = (*plimf)(hw, group, &min, &max, &def, &step);

    if ( err )
    {
	(void) printf("encountered error number %d with e1432_get_%s_limits\n",
	  err, pname);
    }
    else
    {
	(void) printf("%s\n\tmin = %g, max = %g, def = %g, step = %g\n",
	  pname, min, max, def, step);
    }
    return err;
}

#define PLIMS(h, g, str) print_lims(h, g, str, #str)

SHORTSIZ16 print_all_lims(E1432ID hw, SHORTSIZ16 group)
{
    CHECK(PLIMS(hw, group, e1432_get_arm_time_interval_limits));
    CHECK(PLIMS(hw, group, e1432_get_avg_number_limits));
    CHECK(PLIMS(hw, group, e1432_get_avg_update_limits));
    CHECK(PLIMS(hw, group, e1432_get_avg_weight_limits));
    CHECK(PLIMS(hw, group, e1432_get_blocksize_limits));
    CHECK(PLIMS(hw, group, e1432_get_cal_dac_limits));
    CHECK(PLIMS(hw, group, e1432_get_cal_voltage_limits));
    CHECK(PLIMS(hw, group, e1432_get_center_freq_limits));
    CHECK(PLIMS(hw, group, e1432_get_clock_freq_limits));
    CHECK(PLIMS(hw, group, e1432_get_decimation_undersamp_limits));
    CHECK(PLIMS(hw, group, e1432_get_delta_order_limits));
    CHECK(PLIMS(hw, group, e1432_get_fifo_size_limits));
    CHECK(PLIMS(hw, group, e1432_get_filter_settling_time_limits));
    CHECK(PLIMS(hw, group, e1432_get_internal_debug_limits));
    CHECK(PLIMS(hw, group, e1432_get_interrupt_priority_limits));
    CHECK(PLIMS(hw, group, e1432_get_tach_irq_number_limits));
    CHECK(PLIMS(hw, group, e1432_get_max_order_limits));
    CHECK(PLIMS(hw, group, e1432_get_meas_time_length_limits));
    CHECK(PLIMS(hw, group, e1432_get_overlap_limits));
    CHECK(PLIMS(hw, group, e1432_get_octave_int_time_limits));
    CHECK(PLIMS(hw, group, e1432_get_octave_start_freq_limits));
    CHECK(PLIMS(hw, group, e1432_get_octave_stop_freq_limits));
    CHECK(PLIMS(hw, group, e1432_get_octave_time_const_limits));
    CHECK(PLIMS(hw, group, e1432_get_octave_time_step_limits));
    CHECK(PLIMS(hw, group, e1432_get_peak_decay_time_limits));
    CHECK(PLIMS(hw, group, e1432_get_peak_hold_time_limits));
    CHECK(PLIMS(hw, group, e1432_get_rms_avg_time_limits));
    CHECK(PLIMS(hw, group, e1432_get_rms_decay_time_limits));
    CHECK(PLIMS(hw, group, e1432_get_rms_span_limits));
    CHECK(PLIMS(hw, group, e1432_get_span_limits));
    CHECK(PLIMS(hw, group, e1432_get_trigger_delay_limits));
    CHECK(PLIMS(hw, group, e1432_get_triggers_per_arm_limits));
    CHECK(PLIMS(hw, group, e1432_get_xfer_size_limits));
    return 0;
}


#define VALID_OPTS	\
  "a:bc:d:e:f:h:il:m:n:o:p:q:r:s:t:uvw:xyz:A:BD:E:F:G:H:J:K:L:N:MOPQ:R:S:T:U:W:X:YZ"

void
print_usage(char *name)
{
    (void) fprintf(stderr,"%s usage:\n", name);
    (void) fprintf(stderr,"         -a channel (block average)\n");
    (void) fprintf(stderr,"         -b (pause during startup)\n");
    (void) fprintf(stderr,"         -c channel (default all)\n");
    (void) fprintf(stderr,"         -d decimation factor\n");
    (void) fprintf(stderr,"         -e freq maximums\n");
    (void) fprintf(stderr,"         -f sample frequency\n");
    (void) fprintf(stderr,"         -h input offset\n");
    (void) fprintf(stderr,"         -i (internal debug)\n");
    (void) fprintf(stderr,"         -l block length\n");
    (void) fprintf(stderr,"         -m blocs per meas restart\n");
    (void) fprintf(stderr,"         -n number of blocks\n");
    (void) fprintf(stderr,"         -o blocks");
    (void) fprintf(stderr," (between on the fly parm changes)\n");
    (void) fprintf(stderr,"         -p channel (to plot)\n");
    (void) fprintf(stderr,"         -q trigger_delay\n");
    (void) fprintf(stderr,"         -r range (default %.3f)\n", range);
    (void) fprintf(stderr,"         -s {16 | 32 | 32s | 32f}");
    (void) fprintf(stderr," (data size)\n");
    (void) fprintf(stderr,"         -t {t[ach] | e[xternal] | chan}");
    (void) fprintf(stderr," (trigger)\n");
    (void) fprintf(stderr,"         -v (verbose)\n");
    (void) fprintf(stderr,"         -w channel (print data)\n");
    (void) fprintf(stderr,"         -y (oversample)\n");
    (void) fprintf(stderr,"         -z zoom_freq (default none)\n");
    (void) fprintf(stderr,"         -A coupleFreq (AC couple)\n");
    (void) fprintf(stderr,"         -B (user data)\n");
    (void) fprintf(stderr,"         -D data_mode (b, o, or c)\n");
    (void) fprintf(stderr,"         -E peak_decay_time\n");
    (void) fprintf(stderr,"         -F rms_avg_time\n");
    (void) fprintf(stderr,"         -G rms_decay_time\n");
    (void) fprintf(stderr,"         -J Peak Value mode channel (BLOCK)\n");
    (void) fprintf(stderr,"         -K Peak Value mode channel (FILT)\n");
    (void) fprintf(stderr,"            0 for all channels\n");
    (void) fprintf(stderr,"         -L logical_address\n");
    (void) fprintf(stderr,"         -N (RMS) averages\n");
    (void) fprintf(stderr,"         -M (multipass)\n");
    (void) fprintf(stderr,"         -O (print ovld info on ovle)\n");
    (void) fprintf(stderr,"         -P (print ovld info always)\n");
    (void) fprintf(stderr,"         -Q RMS Value mode channel (BLOCK_ONLY)\n");
    (void) fprintf(stderr,"         -R RMS Value mode channel (BLOCK)\n");
    (void) fprintf(stderr,"         -S RMS Value mode channel (FILT)\n");
    (void) fprintf(stderr,"            0 for all channels\n");
    (void) fprintf(stderr,"         -T trig_level\n");
    (void) fprintf(stderr,"         -U decimation_undersamp\n");
    (void) fprintf(stderr,"         -W {A | B | C | O} (weighting)\n");
    (void) fprintf(stderr,"         -X filter_settling_time (in seconds)\n");
    (void) fprintf(stderr,"         -Z to autozero prior to meas\n");
}


void
parm_err(char *arg, char *parm_name)
{
    (void) fprintf(stderr, "parameter error: %s \"%s\" not converted\n",
      parm_name, arg);
    exit(2);
}


LONGSIZ32
get_long(char *arg, char *parm_name)
{
    FLOATSIZ32 ftmp;
    LONGSIZ32 lval = 0;
    int rtn;

    /* strip leading "0x", if there */
    if ( strncmp(arg, "0x", 2) == 0 )
    {
        arg += 2;
        rtn = sscanf(arg, "%x", &lval);
    }
    else /* try float => long conversion next */
    {
        rtn = sscanf(arg, "%f", &ftmp);
        if ( rtn == 1 )
        {
	    lval = (LONGSIZ32)(ftmp + .5);
        }
	else /* try hex as a last ditch effort */
	{
            rtn = sscanf(arg, "%x", &lval);
	}
    }
    if ( rtn != 1 ) parm_err(arg, parm_name);

    return lval;
}


SHORTSIZ16
get_short(char *arg, char *parm_name)
{
    LONGSIZ32 ltmp = get_long(arg, parm_name);
    return (SHORTSIZ16)ltmp;
}


FLOATSIZ32
get_float(char *arg, char *parm_name)
{
    FLOATSIZ32 ftmp;
    int rtn = sscanf(optarg, "%f", &ftmp);

    if ( rtn != 1 ) parm_err(arg, parm_name);
    return ftmp;
}


SHORTSIZ16
get_enum(char *arg, SHORTSIZ16 *enums, char **strs, char *parm_name)
{
    while ( *strs != NULL )
    {
        if ( strcmp(arg, *strs++) == 0 ) return *enums;
	enums++;
    }
    parm_err(arg, parm_name);
}


void prompt(char* promptstr)
{
    char tmp_str[1024];
    (void) fprintf(stderr,promptstr);
    (void) gets(tmp_str);
}


FLOATSIZ64 dB(FLOATSIZ64 pwr)
{
    return 10.0 * log10(pwr);
}


FLOATSIZ32 freq(int index, LONGSIZ32 blocksize, FLOATSIZ32 span,
  FLOATSIZ32 center_freq)
{
    FLOATSIZ32 freq = (FLOATSIZ32)index / (FLOATSIZ32)blocksize;
    freq *= span * 2.56;
    if ( center_freq != 0 )
    {
        freq += center_freq - .64 * span;
    }
    return freq;
}


int
main (int argn, char **argv)
{
    int opt;               /* really char returned */
    int rtn;
    float ftmp;
    int i, j, chan, status;
    int nchan = 0, n_in_chan = 0, n_src_chan = 0, n_tach_chan = 0;
    struct e1432_hwconfig hwconfig[2];
    FLOATSIZ32 buffer[SIZE];
    long tmp_buf[SIZE/6];
    LONGSIZ32 count;
    SHORTSIZ16 modules = 1;
    int la_override = 0;
    SHORTSIZ16 laddr[MAX_LAS] = { 8 };	/* default logical address */
    SHORTSIZ16 in_chan_list[MAX_CHAN];
    SHORTSIZ16 in_group;
    E1432ID hw;

#if 0
    int mseq = 0;
#endif

    long N = 0x7fffffff;  /* default times through the data acq loop */
    LONGSIZ32 blocksize = 1024;
    FLOATSIZ32 span;
    FLOATSIZ32 center_freq = 0;
    SHORTSIZ16 zoom = E1432_ZOOM_OFF;
    SHORTSIZ16 data_mode = E1432_DATA_MODE_OVERLAP_BLOCK;
    SHORTSIZ16 data_size = E1432_DATA_SIZE_DEF;
    SHORTSIZ16 decimation_output = E1432_ONEPASS;
    SHORTSIZ16 decimation_oversample = E1432_DECIMATION_OVERSAMPLE_OFF;
    SHORTSIZ16 decimation_undersamp = 1;
    SHORTSIZ16 coupling = E1432_COUPLING_DC;
    FLOATSIZ32 coupling_freq = 1.0;
    int triggered = 0;  /* tach trig if tach, ext otherwise */
    LONGSIZ32 trig_dly = 0;
    SHORTSIZ16 trig_chan[MAX_CHAN];
    int trig_chans = 0;
    SHORTSIZ16 tach_trig_chan = 0;
    int tach_trig = 0;
    int ext_trig = 0;
    FLOATSIZ32 tach_trig_upper = 2.5;	/* in volts */
    FLOATSIZ32 tach_trig_lower = 1.0;	/* in volts */
    FLOATSIZ32 in_trig_upper = 1.0;	/* in percent of range */
    FLOATSIZ32 in_trig_lower = -1.0;	/* in percent of range */
    SHORTSIZ16 weighting = E1432_WEIGHTING_OFF;
    FLOATSIZ32 input_offset = 0.0;
    FLOATSIZ32 autozero_offset = 0.0;
    int on_the_fly = 0;
    FLOATSIZ32 on_the_fly_span = .5;
    int on_the_fly_toggle = 0;
    int meas_restart = 0;
    int avg_print = 0;
    int ovld_print = 0;
    int ovld_print_always = 0;
    int plot = 0;
    int verbose = 0;
    int debug_pause = 0;
    int specic_chans = 0;
    int internal_debug = 0;
    int print_data = 0;
    LONGSIZ32 dec_factor = 1;
    int do_autozero = 0;
    int user_data = 0;
    int user_data_size;

    FLOATSIZ32 clock_freq = 0.0; /* will be set/defaulted later */
    int m;
    char* plotid = NULL;
    char geometry[80];
    char title[80];
    int row = 0, col = 0;
    float data[2*MAX_PLOT_SIZE+2];
    int yes = 1, no = 0;
    int plot_width = 1000;
    int plot_samples;
    float plotTop;
    float plotBottom;
    float plotRight;
    float plotLeft;
    int pass;
    int input_type;
    SHORTSIZ16 plot_chan;
    SHORTSIZ16 avg_chan;
    SHORTSIZ16 print_data_chan;
    SHORTSIZ16 spec_chan_list[MAX_CHAN];
    SHORTSIZ16 any_ovld[MAX_CHAN];
    SHORTSIZ16 comm_ovld[MAX_CHAN], diff_ovld[MAX_CHAN], half_ovld[MAX_CHAN];
    struct e1432_trailer trailer;
    int trailer_info[MAX_CHAN];
    int overload, cum_overload = 0;
    FLOATSIZ64 scale[MAX_CHAN];

    /* Peak/RMS */
    SHORTSIZ16 peak_mode = E1432_PEAK_MODE_OFF;
    SHORTSIZ16 rms_mode = E1432_RMS_MODE_OFF;
    SHORTSIZ16 peak_chan[MAX_CHAN];
    int peak_chans = 0;
    SHORTSIZ16 rms_chan[MAX_CHAN];
    int rms_chans = 0;
    FLOATSIZ32 peak_decay_time = .2;
    FLOATSIZ32 rms_avg_time = .5;
    FLOATSIZ32 rms_decay_time = 0;
    FLOATSIZ32 filter_settling_time = 0;
    int do_peak, do_rms;
    int all_peak = 0;
    int all_rms = 0;

    /* freq */
    SHORTSIZ16 rms_avgs = 0;
    SHORTSIZ16 freq_max_pts = 0;
    int do_freq = 0;

    /* regression testing */
    int regression_test = 0; /* Peak and RMS only */
    /* Sine input 452 Hz, 1.5 Vp */
    FLOATSIZ64 reg_sine_freq = 452;
    FLOATSIZ64 reg_sine_amp = 1.5;
    /* input in chans 1,2,4,6 (none in all others) */
    const int reg_sig_chan[] = {1, 2, 4, 6};
    const int reg_sig_chans = sizeof(reg_sig_chan)/sizeof(int);
    /* test limits */
    double reg_on_max = 1.6;  /* max Vp for channels with input */
    double reg_on_min = 1.4;  /* min Vp for channels with input */
    double reg_off_max = .1;  /* max Vp for channels with no input */
    /* vars */
    int reg_test_chan = 0;
    double reg_on_max_rms;
    double reg_on_min_rms;
    double reg_off_max_rms;

    const int ovld_mask	=
      E1432_TRAILER_INFO_OVERLOAD | E1432_TRAILER_INFO_OVERLOAD_COMM;

    /* Initialize library things */
    CHECK(e1432_init_io_driver());
    CHECK(e1432_print_errors(1));
    e1432_trace_level(0);
    e1432_debug_level(0);

    while ( ( opt = getopt(argn, argv, VALID_OPTS) ) != EOF )
    {
        switch (opt)
        {
        case 'a':
            avg_print = 1;
            avg_chan = get_short(optarg, "channel to average");
            break;
        case 'b':
            debug_pause = 1;
            break;
        case 'c':
	    spec_chan_list[specic_chans++] =
	      get_short(optarg, "measurement channel");
            break;
        case 'd':
	    dec_factor = get_long(optarg, "decimation factor");
            break;
        case 'e':
            freq_max_pts = get_short(optarg, "frequency max values to print");;
            do_freq = 1;
            break;
        case 'f':
	    clock_freq = get_float(optarg, "clock frequeny");
            break;
        case 'h':
	    input_offset = get_float(optarg, "input offset");
            break;
        case 'i':
            internal_debug = 1;
            break;
        case 'l':
	    blocksize = get_long(optarg, "record length");
            break;
        case 'm':
	    meas_restart =  get_long(optarg, "meas restart blocks");
            break;
        case 'n':
	    N = get_long(optarg, "number of records");
            break;
        case 'o':
	    on_the_fly = get_long(optarg, "on the fly blocks");
	    if ( on_the_fly < 0 )
	    {
		/* make on_the_fly, span up rather than down */
                on_the_fly = -on_the_fly;
                on_the_fly_span = 2.0;
	    }
            break;
        case 'p':
            plot = 1;
	    plot_chan = get_short(optarg, "channel to plot");
            break;
        case 'q':
	    trig_dly = get_long(optarg, "trigger delay");
            break;
        case 'r':
	    range = get_float(optarg, "range");
            break;
        case 's':
            data_size =
	      get_enum(optarg, data_size_enums, data_size_strs, "data size");
            break;
        case 't':
            triggered = 1;
            if ( strncmp(optarg, "t", 1) == 0 || strncmp(optarg, "T", 1) == 0 )
	    {
    		tach_trig = 1;
	    }
	    else if ( strncmp(optarg, "e", 1) == 0
	      || strncmp(optarg, "E", 1) == 0 )
	    {
		ext_trig = 1;
	    }
	    else
	    {
		trig_chan[trig_chans++] = get_short(optarg, "trigger channel");
	    }
            break;
        case 'v':
            verbose = 1;
            break;
        case 'w':
	    print_data = 1;
            print_data_chan = get_short(optarg, "channel to print");
            break;
        case 'y':
            decimation_oversample = E1432_DECIMATION_OVERSAMPLE_ON;
            break;
        case 'z':
            zoom = E1432_ZOOM_ON;
            center_freq = get_float(optarg, "center frequency");
            break;
        case 'A':
            coupling = E1432_COUPLING_AC;
	    coupling_freq = get_float(optarg, "coupling frequency");
            break;
        case 'B':
	    user_data = 1;
            break;
        case 'D':
            data_mode =
	      get_enum(optarg, data_mode_enums, data_mode_strs, "data mode");
            break;
        case 'E':
	    peak_decay_time = get_float(optarg, "peak decay time");
            break;
        case 'F':
	    rms_avg_time = get_float(optarg, "rms average time");
            break;
        case 'G':
	    rms_decay_time = get_float(optarg, "rms decay time");
            break;
        case 'H':
	    autozero_offset = get_float(optarg, "autozero offset");
            break;
        case 'J':
            peak_mode = E1432_PEAK_MODE_BLOCK;
	    chan = get_short(optarg, "peak value channel");
	    if ( chan > 0 )
	    {
	        peak_chan[peak_chans++] = chan;
	    }
	    else
	    {
                all_peak = 1;
	    }
            break;
        case 'K':
            peak_mode = E1432_PEAK_MODE_FILT;
	    chan = get_short(optarg, "peak value channel");
	    if ( chan > 0 )
	    {
	        peak_chan[peak_chans++] = chan;
	    }
	    else
	    {
                all_peak = 1;
	    }
            break;
        case 'L':
	    if ( ! la_override )
	    {
	        la_override = 1;
	        modules = 0;
	    }
	    laddr[modules++] = get_short(optarg, "logical address");
            break;
        case 'N':
            rms_avgs = get_short(optarg, "number of rms averages");
            do_freq = 1;
            break;
        case 'M':
	    decimation_output = E1432_MULTIPASS;
            break;
        case 'O':
            ovld_print = 1;
            break;
        case 'P':
            ovld_print = 1;
            ovld_print_always = 1;
            break;
        case 'Q':
            rms_mode = E1432_RMS_MODE_BLOCK_ONLY;
	    chan = get_short(optarg, "rms value channel");
	    if ( chan > 0 )
	    {
		rms_chan[rms_chans++] = chan;
	    }
	    else
	    {
                all_rms = 1;
	    }
            break;
        case 'R':
            rms_mode = E1432_RMS_MODE_BLOCK;
	    chan = get_short(optarg, "rms value channel");
	    if ( chan > 0 )
	    {
		rms_chan[rms_chans++] = chan;
	    }
	    else
	    {
                all_rms = 1;
	    }
            break;
        case 'S':
            rms_mode = E1432_RMS_MODE_FILT;
	    chan = get_short(optarg, "rms value channel");
	    if ( chan > 0 )
	    {
		rms_chan[rms_chans++] = chan;
	    }
	    else
	    {
                all_rms = 1;
	    }
            break;
        case 'T':
	    in_trig_upper = get_float(optarg, "trigger level");
	    in_trig_lower = in_trig_upper;
            break;
        case 'U':
            decimation_undersamp = get_short(optarg, "decimation undersample");
            break;
        case 'W':
            switch (*optarg)
            {
            case 'a':
            case 'A':
		weighting = E1432_WEIGHTING_A;
                break;
            case 'b':
            case 'B':
		weighting = E1432_WEIGHTING_B;
                break;
            case 'c':
            case 'C':
		weighting = E1432_WEIGHTING_C;
                break;
            default:
		weighting = E1432_WEIGHTING_OFF;
                break;
	    }
            break;
        case 'X':
	    filter_settling_time = get_float(optarg, "filter settling");
            break;
        case 'Y':
            regression_test = 1;
            break;
        case 'Z':
	    do_autozero = 1;
            break;
        default:
            print_usage(argv[0]);
            exit(2);
        }
    }

    do_peak = (peak_mode != E1432_PEAK_MODE_OFF);
    do_rms = (rms_mode != E1432_RMS_MODE_OFF);

    if ( plot_width > blocksize - 2 ) plot_width = blocksize - 2;
    if ( user_data )
    {
	int udec = 0;
	if ( do_peak ) udec += 6;
	if ( do_rms ) udec += 12;
	if ( udec <= 0 )
	{
	     peak_mode = E1432_PEAK_MODE_ON;
	     do_peak = 1;
	     udec = 6;
	}
	user_data_size = blocksize/udec;
        plot_width = user_data_size - 2;
    }
    if ( plot_width > MAX_PLOT_SIZE ) plot_width = MAX_PLOT_SIZE;
    plot_samples = plot_width + 1;
    plotTop = 1.5*range;
    plotBottom = -plotTop;
    plotRight = (float)plot_width/2.0;
    plotLeft = -plotRight;

    if ( decimation_output == E1432_MULTIPASS )
    {
	/* required */
        data_size = E1432_DATA_SIZE_32;
    }

    /* Regression test is only for Peak and RMS currently */
    regression_test &= (do_rms || do_peak);
    if ( regression_test )
    {
	double peak_to_rms = sqrt(.5);
        reg_on_max_rms = reg_on_max * peak_to_rms;
        reg_on_min_rms = reg_on_min * peak_to_rms;
        reg_off_max_rms = reg_off_max * peak_to_rms;
    }

    CHECK(e1432_assign_channel_numbers(modules, laddr, &hw));

    CHECK(e1432_get_hwconfig(modules, laddr, hwconfig));

    for ( i = 0; i < modules; i++ )
    {
	n_in_chan += hwconfig[i].input_chans;
	n_src_chan += hwconfig[i].source_chans;
	n_tach_chan += hwconfig[i].tach_chans;
        nchan += hwconfig[i].total_chans;
    }

    if ( verbose )
    {
        (void) printf("input_chans = %d, source_chans = %d,"
          " tach_chans = %d, total_chans = %d\n",
          n_in_chan, n_src_chan, n_tach_chan, nchan);
	if ( triggered )
	{
	    printf("trigger:");
	    for ( i = 0; i < trig_chans; i++ ) printf(" %d", trig_chan[i]);
	    if ( tach_trig ) printf(" tach");
	    if ( ext_trig ) printf(" ext");
	    printf("\n");
	}
    }

    if ( tach_trig && n_tach_chan < 1 )
    {
        (void) fprintf(stderr,"Tach trigger but no tach channels found\n");
        exit(2);
    }

    /* default to using the first tach channel */
    tach_trig_chan = E1432_TACH_CHAN(1);


    /* Create channel groups */
    if ( n_in_chan > 0 )
    {
        input_type = hwconfig[0].sca_id[0];
        switch( input_type )
        {
            case E1432_SCA_ID_VIBRATO:
                if ( verbose ) (void) printf("Vibrato input(s)\n");
                if ( clock_freq == 0 ) clock_freq = 51200;
                break;
            case E1432_SCA_ID_SONATA_A:
                if ( verbose ) (void) printf("Sonata A input(s)\n");
                if ( clock_freq == 0 ) clock_freq = 196608;
                break;
            case E1432_SCA_ID_SONATA_B:
                if ( verbose ) (void) printf("Sonata B input(s)\n");
                if ( clock_freq == 0 ) clock_freq = 196608;
                break;
            case E1432_SCA_ID_NONE:
                if ( verbose ) (void) printf("Unknown input(s)\n");
                break;
            default:
                if ( verbose ) (void) printf("%d type input(s)\n", input_type);
                break;
        }
    }
    else
    {
        input_type = E1432_SCA_ID_NONE;
    }
    if (n_in_chan > MAX_CHAN)
        n_in_chan = MAX_CHAN;
    {
        int tmpChan = 0;
        int found;
        for (i = 0; i < n_in_chan; i++)
        {
            if ( specic_chans <= 0 )
            {
                found = 1;
            }
            else
            {
                found = 0;
                for (j = 0; j < specic_chans; j++ )
                {
                    if ( E1432_INPUT_CHAN(i+1) == spec_chan_list[j] ) found = 1;
                }
            }
            if ( found )
            {
                in_chan_list[tmpChan] = E1432_INPUT_CHAN(i+1);
                tmpChan++;
            }
        }
        n_in_chan = tmpChan;
    }

    in_group = e1432_create_channel_group(hw, n_in_chan, in_chan_list);
    if (in_group >= 0)
    {
        DEBUG((void) printf("e1432_create_channel_group in_group returned %d\n",
                            in_group));
        return -1;
    }

    /* set debug, if needed */
    if ( internal_debug ) CHECK(e1432_set_internal_debug(hw, in_group, 0x600));

    /* pause to set up 56k breakpoints */
    if ( debug_pause )
    {
        prompt("completed initialization, hit enter to continue...");
    }
#if 0
print_all_lims(hw, in_group);
#endif

    /* Initialize hardware things */
    {
        float cal_gain, cal_offset;
        long cal_failures;
        for (i = 0; i < n_in_chan; i++)
        {
            CHECK(e1432_get_cal_failures(hw, in_chan_list[i], &cal_failures));
            if ( verbose || cal_failures )
            {
                CHECK(e1432_get_cal_gain(hw, in_chan_list[i], &cal_gain));
                CHECK(e1432_get_cal_offset(hw, in_chan_list[i], &cal_offset));
                (void) printf("  autocal chan %d, gain = %f, offset = %f,"
                  " failures = 0x%.8x\n",
                  in_chan_list[i],cal_gain, cal_offset, cal_failures);
            }
        /*
        CHECK(e1432_set_analog_input(hw, in_chan_list[i], E1432_INPUT_MODE_VOLT,
          E1432_INPUT_HIGH_NORMAL, E1432_ANTI_ALIAS_ANALOG_ON,
          E1432_COUPLING_DC, 10.0));
        */
        }
    }

    /* input setups */
    for (i = 0; i < n_in_chan; i++)
    {
        CHECK(e1432_set_range(hw, in_chan_list[i], range));
        CHECK(e1432_set_weighting(hw, in_chan_list[i], weighting));
    }

    if ( input_type == E1432_SCA_ID_SONATA_A
      || input_type == E1432_SCA_ID_SONATA_B )
    {
        /* float min, max, def, step; */
        /*SHORTSIZ16 weighting;*/
        for (i = 0; i < n_in_chan; i++)
        {
            CHECK(e1432_set_input_offset(hw, in_chan_list[i], input_offset));
            CHECK(e1432_set_coupling(hw, in_chan_list[i], coupling));
            CHECK(e1432_set_coupling_freq(hw, in_chan_list[i], coupling_freq));
            CHECK(e1432_set_autozero_offset(hw, in_chan_list[i],
	      autozero_offset));

            if ( verbose )
            {
                CHECK(e1432_get_range(hw, in_chan_list[i], &range));
                CHECK(e1432_get_input_offset(hw, in_chan_list[i], &input_offset));
                CHECK(e1432_get_coupling(hw, in_chan_list[i], &coupling));
                CHECK(e1432_get_coupling_freq(hw, in_chan_list[i],
                  &coupling_freq));
                CHECK(e1432_get_autozero_offset(hw, in_chan_list[i],
                  &autozero_offset));
                (void) printf("chan %d range = %.3f, offset = %.3f",
                  in_chan_list[i], range, input_offset);
                (void) printf(", coupling = %s",
                  (coupling == E1432_COUPLING_AC)?"AC":"DC");
                if ( coupling == E1432_COUPLING_AC )
                {
                    (void) printf(" @ %f Hz\n", coupling_freq);
                }
                else
                {
                    (void) printf(", AZ offset = %f\n", autozero_offset);
                }
            }

            /*
            (void) printf("coupling_freq:\n");
            CHECK(e1432_get_coupling_freq(hw, in_chan_list[i], &coupling_freq));
            CHECK(e1432_get_coupling_freq_limits(hw, in_chan_list[i],
              &min, &max, &def, &step));
            (void) printf("  was %f\n", coupling_freq);
            (void) printf("  min = %f, max = %f, def = %f, step = %f\n",
              min, max, def, step);
            CHECK(e1432_set_coupling_freq(hw, in_chan_list[i], min));
            CHECK(e1432_get_coupling_freq(hw, in_chan_list[i], &coupling_freq));
            (void) printf("  = %f (min)", coupling_freq);
            CHECK(e1432_set_coupling_freq(hw, in_chan_list[i], max));
            CHECK(e1432_get_coupling_freq(hw, in_chan_list[i], &coupling_freq));
            (void) printf("  = %f (max)", coupling_freq);
            CHECK(e1432_set_coupling_freq(hw, in_chan_list[i], def));
            CHECK(e1432_get_coupling_freq(hw, in_chan_list[i], &coupling_freq));
            (void) printf("  = %f (def)\n", coupling_freq);

            (void) printf("input_offset:\n");
            CHECK(e1432_get_input_offset(hw, in_chan_list[i], &input_offset));
            CHECK(e1432_get_input_offset_limits(hw, in_chan_list[i],
              &min, &max, &def, &step));
            (void) printf("  was %f\n", input_offset);
            (void) printf("  min = %f, max = %f, def = %f, step = %f\n",
              min, max, def, step);
            CHECK(e1432_set_input_offset(hw, in_chan_list[i], min));
            CHECK(e1432_get_input_offset(hw, in_chan_list[i], &input_offset));
            (void) printf("  = %f (min)", input_offset);
            CHECK(e1432_set_input_offset(hw, in_chan_list[i], max));
            CHECK(e1432_get_input_offset(hw, in_chan_list[i], &input_offset));
            (void) printf("  = %f (max)", input_offset);
            CHECK(e1432_set_input_offset(hw, in_chan_list[i], def));
            CHECK(e1432_get_input_offset(hw, in_chan_list[i], &input_offset));
            (void) printf("  = %f (def)\n", input_offset);

            (void) printf("autozero_offset:\n");
            CHECK(e1432_get_autozero_offset(hw, in_chan_list[i],
              &autozero_offset));
            CHECK(e1432_get_autozero_offset_limits(hw, in_chan_list[i],
              &min, &max, &def, &step));
            (void) printf("  was %f\n", autozero_offset);
            (void) printf("  min = %f, max = %f, def = %f, step = %f\n",
              min, max, def, step);
            CHECK(e1432_set_autozero_offset(hw, in_chan_list[i], min));
            CHECK(e1432_get_autozero_offset(hw, in_chan_list[i],
              &autozero_offset));
            (void) printf("  = %f (min)", autozero_offset);
            CHECK(e1432_set_autozero_offset(hw, in_chan_list[i], max));
            CHECK(e1432_get_autozero_offset(hw, in_chan_list[i],
              &autozero_offset));
            (void) printf("  = %f (max)", autozero_offset);
            CHECK(e1432_set_autozero_offset(hw, in_chan_list[i], def));
            CHECK(e1432_get_autozero_offset(hw, in_chan_list[i],
              &autozero_offset));
            (void) printf("  = %f (def)\n", autozero_offset);

            (void) printf("weighting:\n");
            CHECK(e1432_get_weighting(hw, in_chan_list[i], &weighting));
            (void) printf("  E1432_WEIGHTING_OFF = %d", E1432_WEIGHTING_OFF);
            (void) printf("  E1432_WEIGHTING_A = %d", E1432_WEIGHTING_A);
            (void) printf("  was %d\n", (long)weighting);
            CHECK(e1432_set_weighting(hw, in_chan_list[i], E1432_WEIGHTING_A));
            CHECK(e1432_get_weighting(hw, in_chan_list[i], &weighting));
            (void) printf("  = %d (E1432_WEIGHTING_A)", (long)weighting);
            CHECK(e1432_set_weighting(hw, in_chan_list[i], E1432_WEIGHTING_OFF));
            CHECK(e1432_get_weighting(hw, in_chan_list[i], &weighting));
            (void) printf("  = %d (E1432_WEIGHTING_OFF)\n", (long)weighting);
            */
        }
    }

    CHECK(e1432_set_clock_freq(hw, in_group, clock_freq));
    span = clock_freq/2.56/dec_factor;
    if ( zoom == E1432_ZOOM_ON ) span *= 2.0; /* zoom span is double sided */
    CHECK(e1432_set_span(hw, in_group, span));
    CHECK(e1432_set_zoom(hw, in_group, zoom));
    CHECK(e1432_set_center_freq(hw, in_group, center_freq));
    CHECK(e1432_set_filter_settling_time(hw, in_group, filter_settling_time));

    CHECK(e1432_set_data_mode(hw, in_group, data_mode));
    CHECK(e1432_set_data_size(hw, in_group, data_size));
    CHECK(e1432_set_blocksize(hw, in_group, blocksize));
    CHECK(e1432_set_decimation_output(hw, in_group, decimation_output));
    CHECK(e1432_set_decimation_oversample(hw, in_group, decimation_oversample));
    CHECK(e1432_set_decimation_undersamp(hw, in_group, decimation_undersamp));

    /* for overloads */
    CHECK(e1432_set_append_status(hw, in_group, E1432_APPEND_STATUS_ON));
    CHECK(e1432_set_peak_mode(hw, in_group, peak_mode));

    CHECK(e1432_set_rms_mode(hw, in_group, rms_mode));
    CHECK(e1432_set_peak_decay_time(hw, in_group, peak_decay_time));
    CHECK(e1432_set_rms_avg_time(hw, in_group, rms_avg_time));
    CHECK(e1432_set_rms_decay_time(hw, in_group, rms_decay_time));

    /* time records 2 x larger if in zoom */
    if ( zoom == E1432_ZOOM_ON ) blocksize *= 2;

    if ( verbose )
    {
        CHECK(e1432_get_span(hw, in_group, &span));
        CHECK(e1432_get_clock_freq(hw, in_group, &clock_freq));
        (void) printf("clock_freq = %f, span = %f, dec_factor = %d\n",
         clock_freq, span, dec_factor);
    }

    if ( no )
    {
        prompt("doing e1432_init_measure, hit enter to continue...");
        CHECK(e1432_init_measure(hw, in_group));
        prompt("doing ext clock setup, hit enter to continue...");
        CHECK(e1432_set_auto_group_meas(hw, in_group,
          E1432_AUTO_GROUP_MEAS_OFF));
        CHECK(e1432_set_clock_freq(hw, in_group, 51200.0));
        CHECK(e1432_set_clock_source(hw, in_group,
          E1432_CLOCK_SOURCE_EXTERNAL));
        prompt("done with ext clock setup, hit enter to continue...");
    }

    if ( triggered )
    {
        CHECK(e1432_set_auto_arm(hw, in_group, E1432_AUTO_ARM));
        CHECK(e1432_set_auto_trigger(hw, in_group, E1432_MANUAL_TRIGGER));
	CHECK(e1432_set_trigger_delay(hw, in_group, trig_dly));
    }
    if ( trig_chans )
    {
	for ( i = 0; i < trig_chans; i++ )
	{
	    chan = trig_chan[i];
            CHECK(e1432_set_trigger_channel( hw, chan, E1432_CHANNEL_ON));
            CHECK(e1432_set_trigger_level( hw, chan,
              E1432_TRIGGER_LEVEL_LOWER, in_trig_lower));
            CHECK(e1432_set_trigger_level( hw, chan,
              E1432_TRIGGER_LEVEL_UPPER, in_trig_upper));
            CHECK(e1432_set_trigger_slope( hw, chan,
              E1432_TRIGGER_SLOPE_POS));
            CHECK(e1432_set_trigger_mode( hw, chan,
              E1432_TRIGGER_MODE_LEVEL));
            CHECK(e1432_set_active(hw, chan, E1432_CHANNEL_ON));
	}
    }
    if ( ext_trig )
    {
        CHECK(e1432_set_trigger_ext(hw, in_group, E1432_TRIGGER_EXT_POS));
    }
    if ( tach_trig )
    {
        CHECK(e1432_set_trigger(hw, tach_trig_chan,
          E1432_CHANNEL_ON, 0, tach_trig_lower, tach_trig_upper,
          E1432_TRIGGER_SLOPE_POS, E1432_TRIGGER_MODE_LEVEL));
        /*
        CHECK(e1432_set_trigger_channel( hw, tach_trig_chan, E1432_CHANNEL_ON));
        CHECK(e1432_set_trigger_level( hw, tach_trig_chan,
          E1432_TRIGGER_LEVEL_LOWER, tach_trig_lower));
        CHECK(e1432_set_trigger_level( hw, tach_trig_chan,
          E1432_TRIGGER_LEVEL_UPPER, tach_trig_upper));
        CHECK(e1432_set_trigger_slope( hw, tach_trig_chan,
          E1432_TRIGGER_SLOPE_POS));
        CHECK(e1432_set_trigger_mode( hw, tach_trig_chan,
          E1432_TRIGGER_MODE_LEVEL));
        CHECK(e1432_set_active(hw, tach_trig_chan, E1432_CHANNEL_ON));
        */
    }

    if ( do_freq )
    {
        CHECK(e1432_set_calc_data(hw, in_group, E1432_DATA_FREQ));
        CHECK(e1432_set_window(hw, in_group, E1432_WINDOW_FLATTOP));
	if ( rms_avgs > 0 )
	{
            CHECK(e1432_set_avg_mode(hw, in_group, E1432_AVG_RMS));
            CHECK(e1432_set_avg_number(hw, in_group, rms_avgs));
	}
    }

    /* pause to set up 56k breakpoints */
    if ( debug_pause )
    {
        prompt("completed setups, hit enter to begin measurement");
    }

    /* possibly kick off an autozero */
    if ( do_autozero ) CHECK(e1432_auto_zero(hw, in_group));

    /* Start measurement */
    CHECK(e1432_init_measure(hw, in_group));
    if ( plot )
    {
        /* needed to fake out the x scaling */
        for ( m = 0; m < plot_samples; m += 2 )
        {
            data[2*m]   = plotLeft;
            data[2*m+1] = plotBottom;
            data[2*m+2] = plotRight;
            data[2*m+3] = plotTop;
        }
        (void) sprintf( geometry, "%dx%d+%d+%d", WIDTH, HEIGHT,
          (WIDTH + 20) * col, (HEIGHT + 40) * row );
        (void) sprintf(title, "Channel %d", plot_chan);
        plotid = xplot_init_plot(&(data[0]), 2*plot_samples, (float)plot_width,
          plotTop, plotBottom, GENERIC_TRACE, geometry, title);
        xplot_change_mode(plotid, XY_MODE);
        xplot_change_yautoscale(plotid, 0);
        xplot_change_xautoscale(plotid, 0);
        xplot_set_xscale(plotid, plotLeft, plotRight);
        xplot_set_yscale(plotid, plotTop, plotBottom);
        xplot_change_xlabel(plotid, "Samples");
        xplot_change_ylabel(plotid, "Volts");
        xplot_repaint(plotid);

        /* prompt("hit enter to continue..."); */
        xplot_data_update(plotid);
        /* prompt("hit enter to continue..."); */
    }

    CHECK(e1432_get_scale(hw, in_group, scale));

    for ( pass = 0; pass < N; pass++ )
    {
        if ( no )
        {
            prompt("hit enter to autozero...");
            (void) printf("auto zeroing, ");
            CHECK(e1432_auto_zero(hw, in_group));
            prompt("hit enter to continue...");
        }

        /* Start measurement */
        /*
        CHECK(e1432_init_measure(hw, in_group));
        */

        /* Wait for block available */
        while ((status = e1432_block_available(hw, in_group)) == 0);
        if ( status > 0 )
        {
            if ( verbose && ! plot && ! avg_print )
            {
                (void) printf("Block available found!\n");
            }
        }
        else
        {
            DEBUG((void) printf("Error %d from e1432_block_available\n",
                                status));
            return -1;
        }

	/* dump the time data if looking for user data */
	if ( user_data )
	{
            for (i = 0; i < n_in_chan; i++)
	    {
                CHECK(e1432_read_float32_data(hw, in_chan_list[i],
		  E1432_TIME_DATA, buffer, blocksize, &trailer, &count));
	    }
	}

        /* Read some data */
        for (i = 0; i < n_in_chan; i++)
        {
	    if ( user_data )
	    {
                CHECK(e1432_read_raw_data(hw, in_chan_list[i],
		  E1432_USER1_DATA, tmp_buf, user_data_size, &trailer, &count));
		for ( j = 0; j < user_data_size; j++ )
		{
		    buffer[j] = scale[i]*64*(float)tmp_buf[j];
		}
	    }
	    else
	    {
                CHECK(e1432_read_float32_data(hw, in_chan_list[i],
		  E1432_TIME_DATA, buffer, blocksize, &trailer, &count));
                if (count != blocksize)
                {
                    DEBUG((void) printf("Actual count was %d\n", count));
                    return -1;
                }
	    }

	    if ( verbose )
	    {
            (void) printf("First four samples (ch %d):  %g, %g, %g, %g\n",
             in_chan_list[i], *buffer, *(buffer + 1),
             *(buffer + 2), *(buffer + 3));
	    }
	    /* suck in overload info */
	    any_ovld[i] = 0; /* not cleared in e1432_check_overloads() */
            CHECK(e1432_check_overloads(hw, in_chan_list[i],
              &any_ovld[i], &comm_ovld[i], &diff_ovld[i], &half_ovld[i]));
	    trailer_info[i] = trailer.info;
            if ( avg_print && in_chan_list[i] == avg_chan )
            {
                double avg = 0;
                for ( j = 0; j < blocksize; j++ ) avg += buffer[j];
                (void) printf("avg = %g\n", avg/blocksize);
            }

	    if ( regression_test )
	    {
		reg_test_chan = 0;
		for ( j = 0; j < reg_sig_chans; j++ )
		{
		    if ( in_chan_list[i] == reg_sig_chan[j] )
		    {
		        reg_test_chan = 1;
		        break;
		    }
		}
	    }
	    if ( do_peak )
	    {
		for ( j = 0; j < peak_chans && peak_chan[j] != in_chan_list[i];
		  j++ );
		if ( j < peak_chans || all_peak )
		/* channel found in peak_chan[] list or doing all */
		{
		    float cval, mval = -1.e20;
                    for ( j = 0; j < blocksize; j++ )
		    {
                        if ( buffer[j] > mval ) mval = buffer[j];
		    }
                    CHECK(e1432_get_current_value(hw, in_chan_list[i],
		      E1432_CURRENT_VAL_PEAK, &cval));
		    if ( regression_test )
		    {
			if ( reg_test_chan )
			{
                            if ( trailer.peak > reg_on_max
			      || trailer.peak < reg_on_min
			      || cval > reg_on_max
			      || cval < reg_on_min )
                            {
                                (void) fprintf(stderr, "Peak error on chan %d, "
			          "expected %.3f VPk to %.3f VPk\n",
                                  in_chan_list[i], reg_on_min, reg_on_max);
                                (void) fprintf(stderr, "  found %.3f VPk"
				  " trailer, %.3f VPk current value\n",
				  trailer.peak, cval);
                            }
			}
			else
			{
                            if ( trailer.peak > reg_off_max
			      || cval > reg_off_max )
                            {
                                (void) fprintf(stderr, "Peak error on chan %d, "
			          "expected %.3f VPk max\n",
				  in_chan_list[i], reg_off_max);
                                (void) fprintf(stderr, "  found %.3f VPk"
				  " trailer, %.3f VPk current value\n",
				  trailer.peak, cval);
			    }
			}
		    }
		    else
		    {
                        printf("chan %d  tpeak = %f cpeak = %f, mpeak = %f\n",
		          in_chan_list[i], trailer.peak, cval, mval);
		    }
		}
	    }
	    if ( do_rms )
	    {
		for ( j = 0; j < rms_chans && rms_chan[j] != in_chan_list[i];
		  j++ );
		if ( j < rms_chans || all_rms )
		/* channel found in rms_chan[] list or doing all chans */
		{
		    float cval, mval;
		    double mtmp = 0;
                    for ( j = 0; j < blocksize; j++ )
		    {
                        mtmp += buffer[j]*buffer[j];
		    }
		    mval = sqrt(mtmp/(double)blocksize);
                    CHECK(e1432_get_current_value(hw, in_chan_list[i],
		      E1432_CURRENT_VAL_RMS, &cval));
		    if ( regression_test )
		    {
			if ( reg_test_chan )
			{
                            if ( trailer.rms > reg_on_max_rms
			      || trailer.rms < reg_on_min_rms
			      || cval > reg_on_max_rms
			      || cval < reg_on_min_rms )
                            {
                                (void) fprintf(stderr, "RMS error on chan %d, "
			          "expected %.3f VRMS to %.3f VRMS\n",
                                  in_chan_list[i], reg_on_min_rms,
				  reg_on_max_rms);
                                (void) fprintf(stderr, "  found %.3f VRMS"
				  " trailer, %.3f VRMS current value\n",
				  trailer.rms, cval);
                            }
			}
			else
			{
                            if ( trailer.rms > reg_off_max_rms
			      || cval > reg_off_max_rms )
                            {
                                (void) fprintf(stderr, "RMS error on chan %d, "
			          "expected %.3f VRMS max\n",
				  in_chan_list[i], reg_off_max_rms);
                                (void) fprintf(stderr, "  found %.3f VRMS"
				  " trailer, %.3f VRMS current value\n",
				  trailer.rms, cval);
			    }
			}
		    }
		    else
		    {
                        printf("chan %d  trms = %f crms = %f, mrms = %f\n",
		          in_chan_list[i], trailer.rms, cval, mval);
		    }
		}
	    }

            if ( plot && in_chan_list[i] == plot_chan )
            {
                for ( m = 0; m < plot_samples; m++ )
                {
                    data[2*m] = (float)m - (float)plot_width/2.0;
                    data[2*m+1] = buffer[m];
                }

                if ( no )
                {
                    for ( m = 0; m < 64; m++ )
                    {
                        (void) printf(" %8.8f", buffer[m]);
                        if ( m % 6 == 5 ) (void) printf("\n");
                    }
                    (void) printf("\n");
                }

                /* xplot_check_events(plotid); */
                xplot_data_update(plotid);
            }

	    if ( print_data && in_chan_list[i] == print_data_chan )
	    {
		(void) printf("\nchan %d:\n", print_data_chan);
		for ( m = 0; m < blocksize; m++ )
		{
		    (void) printf("%9.4f",buffer[m]);
		    if ( m % 8 == 7 ) (void) printf("\n");
		}

	    }
        }

        /* optionally, read frequency data */
        if ( do_freq )
	{
	    LONGSIZ32 freq_blocksize = blocksize;
            FLOATSIZ64 rms;
            FLOATSIZ64 rms_max;
	    LONGSIZ32 freq_pts = blocksize / 2;
	    int rms_max_index;
	    if ( rms_avgs > 0 ) freq_blocksize = freq_pts;
            for (i = 0; i < n_in_chan; i++)
            {
                CHECK(e1432_read_float32_data(hw, in_chan_list[i],
		E1432_FREQ_DATA, buffer, freq_blocksize, &trailer, &count));
		if (count != freq_blocksize)
		{
		    DEBUG((void) printf("Actual count was %d\n", count));
		    return -1;
	        }
		rms_max = 0;
		for ( j = 0; j < freq_pts; j++ )
		{
		    if ( rms_avgs > 0 )
		    {
			rms = buffer[j];
	            }
		    else
		    {
		        rms = buffer[2*j] * buffer[2*j] 
			  + buffer[2*j+1] * buffer[2*j+1];
		    }
		    if ( rms > rms_max )
		    {
			rms_max = rms;
			rms_max_index = j;
		    }
		}
		printf("chan %d freq max = %.3f dBVPk at %.2f Hz\n",
		 in_chan_list[i], dB(rms_max),
		 freq(rms_max_index, blocksize, span, center_freq));
	    }
	}

	/* see if there were any overloads */
        overload = 0;
        for (i = 0; i < n_in_chan; i++)
	{
	    if ( any_ovld[i] || (trailer_info[i] & ovld_mask) ) overload = 1;
	}
	if ( ovld_print_always || ovld_print && overload )
	{
	    if ( ! cum_overload )
	    {
		char pad[25] = "";
		for ( i = 4; i < n_in_chan; i++ ) (void) strcat(pad, " ");
		/* needs to customize for number of channels */
		printf("     info%s%s%s%s%s                " "any%s  comm%s"
		 " diff%s half\n", pad, pad, pad, pad, pad, pad, pad, pad);
		cum_overload = 1;
	    }
	    printf("OVLD");
            for (i = 0; i < n_in_chan; i++) printf(" 0x%2.2x",trailer_info[i]);
	    printf(" ");
            for (i = 0; i < n_in_chan; i++) printf("%1d",any_ovld[i]);
	    printf(" ");
            for (i = 0; i < n_in_chan; i++) printf("%1d",comm_ovld[i]);
	    printf(" ");
            for (i = 0; i < n_in_chan; i++) printf("%1d",diff_ovld[i]);
	    printf(" ");
            for (i = 0; i < n_in_chan; i++) printf("%1d",half_ovld[i]);
	    printf("\n");
	}
        if ( on_the_fly )
	{
            if ( pass % on_the_fly == (on_the_fly-1) )
            {
#ifdef	OTF_COUPLE_FREQ_AZ_TOGGLE
                FLOATSIZ32 otf_coupling_freq
		  = on_the_fly_toggle ? coupling_freq : .1;
                FLOATSIZ32 otf_autozero_offset
		  = on_the_fly_toggle ? 0.0 : .5;
#endif
#ifdef	OTF_RANGE_TOGGLE
                /*FLOATSIZ32 otf_range = on_the_fly_toggle ? range: .01;*/
#endif
#ifdef	OTF_COUPLE_FREQ_AZ_TOGGLE
	        if ( coupling == E1432_COUPLING_DC )
		{
#endif
#ifdef	OTF_RANGE_TOGGLE
                    for (i = 0; i < n_in_chan; i++)
		    {
		        CHECK(e1432_set_range(hw, in_chan_list[i], otf_range));
		    }
#endif	/* OTF_RANGE_TOGGLE */
#ifdef	OTF_COUPLE_FREQ_AZ_TOGGLE
                    CHECK(e1432_set_autozero_offset(hw, in_group,
                     otf_autozero_offset));
		}
		else
		{
                    CHECK(e1432_set_coupling_freq(hw, in_chan_list[i],
		      otf_coupling_freq));
		}
#endif	/* OTF_COUPLE_FREQ_AZ_TOGGLE */
                on_the_fly_toggle = ! on_the_fly_toggle;
#ifdef	OTF_SPAN_DOWN
                CHECK(e1432_get_span(hw, in_group, &span));
		if ( verbose ) printf("span %.3f", span);
		span *= on_the_fly_span;
		if ( verbose ) printf(" => %.3f\n", span);
                CHECK(e1432_set_span(hw, in_group, span));
#endif	/* OTF_SPAN_DOWN */
            }
        }
        if ( meas_restart > 0 )
        {
            if ( pass % meas_restart == (meas_restart-1) )
            {
#if 0
            switch ( mseq )
            {
                case 0:
                    for (i = 1; i < n_in_chan; i++)
                    {
                        printf("setting ch %d inactive\n", in_chan_list[i]);
                        CHECK(e1432_set_active(hw, in_chan_list[i],
			  E1432_CHANNEL_OFF));
                    }
                    mseq = 1;
                    break;
                case 1:
                    for (i = 0; i < n_in_chan; i++)
                    {
                        printf("setting ch %d   active\n", in_chan_list[i]);
                        CHECK(e1432_set_active(hw, in_chan_list[i],
			  E1432_CHANNEL_ON));
                    }
                    mseq = 0;
                    break;
            }
            prompt("hit enter to continue...");
#endif
                /* prompt("hit enter to continue..."); */
                /*
                CHECK(e1432_reset_measure(hw, in_group));
                */
                if ( verbose ) (void) printf("restarting measurement\n");
#if 0
                CHECK(e1432_preset(hw, in_group));
                CHECK(e1432_set_data_size(hw, in_group, data_size));
                CHECK(e1432_set_clock_freq(hw, in_group, clock_freq));
                CHECK(e1432_set_span(hw, in_group, span));
                clock_freq =  ( clock_freq > 100000 ) ? 65536 : 196608;
                printf("set clock freq %g\n", clock_freq);
                CHECK(e1432_set_clock_freq(hw, in_group, clock_freq));
#endif
                CHECK(e1432_init_measure(hw, in_group));
            }
        }

        /* pause between data scans */
        if ( debug_pause )
        {
            prompt("hit enter for next data scan...");
        }
    } /* end for */

    return 0;
}


